/*
 * Copyright (C) 2015 Jiada Wang <jiada_wang@mentor.com>
 * Copyright (C) 2015 Mentor Graphics
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <arpa/inet.h>
#include <stdbool.h>
#include <sys/fcntl.h>
#include <time.h>
#include <pthread.h>
#include <getopt.h>
#include <inttypes.h>

#if HAVE_SYSTEMD
#include <systemd/sd-daemon.h>
#endif

#include "dgram_service.h"
#include "inc_wdt.h"

static int inc_wdt_establish_scc_inc_com(struct inc_wdt_data *data,
					char *local_addr, char *remote_addr)
{
	int ret = 0;
	struct sockaddr_in local, remote;

	data->sockfd = socket(AF_INC, SOCK_STREAM, 0);
	if (data->sockfd < 0)
		return -errno;

	data->dgram = dgram_init(data->sockfd, INC_WDT_DGRAM_MAX, AF_INC);
	if (!data->dgram) {
		ret = -EINVAL;
		goto out_sock_shutdown;
	}

	local.sin_family = AF_INET;
	local.sin_addr.s_addr = inet_addr(local_addr);
	local.sin_port = htons(WDG_PORT);

	ret = bind(data->sockfd, (struct sockaddr*) &local, sizeof(local));
	if (ret) {
		ret = -errno;
		goto out_dgram_exit;
	}

	remote.sin_family = AF_INET;
	remote.sin_addr.s_addr = inet_addr(remote_addr);
	remote.sin_port = htons(WDG_PORT);

	ret = connect(data->sockfd, (struct sockaddr *) &remote, sizeof(remote));
	if (ret) {
		ret = -errno;
		goto out_dgram_exit;
	}

	return 0;

out_dgram_exit:
	dgram_exit(data->dgram);

out_sock_shutdown:
	shutdown(data->sockfd, SHUT_RDWR);
	close(data->sockfd);

	return ret;
}

static int inc_wdt_send_wdg_msg(struct inc_wdt_data *data)
{
	int ret = 0;
	char buffer[1] = { INC_WDT_MSGID_WDGS_WDG_C_WATCHDOG };

	if (!data->response_received) {
		data->command_prevented = true;
		return -EBUSY;
	}

	ret = dgram_send(data->dgram, buffer, sizeof(buffer));
	if (ret < 0) {
		fprintf(stderr, "dgram_send failed with %d\n", ret);
		return ret;
	}

	data->command_prevented = false;
	data->response_received = false;

#if HAVE_SYSTEMD
	if (data->notify_sd)
		sd_notify(0, "WATCHDOG=1");
#endif

	return ret;
}

static void inc_wdt_on_inc_msg_received(struct inc_wdt_data *data)
{
	if (data->recv_buffer[0] != INC_WDT_MSGID_WDG_WDGS_R_WATCHDOG)
		return;

	pthread_mutex_lock(&data->lock);
	data->response_received = true;
	if (data->command_prevented)
		inc_wdt_send_wdg_msg(data);
	pthread_mutex_unlock(&data->lock);
}

static int inc_wdt_recv_inc_msg(struct inc_wdt_data *data)
{
	int ret = 0;

	ret = dgram_recv(data->dgram, data->recv_buffer,
			INC_WDT_RECEIVE_BUFFER_LEN);
	if ((ret < 0) && (ret != -EAGAIN))
		fprintf(stderr, "dgram_recv failed with %d\n", ret);

	return ret;
}

static void *inc_wdt_rx_task(void *arg)
{
	struct inc_wdt_data *data = (struct inc_wdt_data *)arg;

	do {
		if (inc_wdt_recv_inc_msg(data) > 0)
			inc_wdt_on_inc_msg_received(data);
	} while(1);

	return NULL;
}

static int inc_wdt_init(struct inc_wdt_data *data, char *local_addr,
			char *remote_addr, uint64_t usec,
			uint64_t usec_retry, bool notify)
{
	int ret = 0;
	int timeval_sec = 0;
	long timeval_nsec = 0;

	/* initilize imx2 wdt ping count */
	data->ping_count = 0;

	/* set timer */
	timeval_sec = usec / 1000000;
	timeval_nsec = (long)(usec % 1000000) * (1000L);
	if ((timeval_sec == 0) && (timeval_nsec == 0)) {
		fprintf(stderr, "invalid timer interval value\n");
		return -EINVAL;
	}

	data->timer.tv_sec = timeval_sec;
	data->timer.tv_nsec = timeval_nsec;

	/* set retry timer */
	timeval_sec = usec_retry / 1000000;
	timeval_nsec = (long)(usec_retry % 1000000) * (1000L);
	if ((timeval_sec == 0) && (timeval_nsec == 0)) {
		fprintf(stderr, "invalid retry timer interval value\n");
		return -EINVAL;
	}

	data->timer_retry.tv_sec = timeval_sec;
	data->timer_retry.tv_nsec = timeval_nsec;

	/* set to true for the initial INC watchdog message transmission */
	data->response_received = true;
	data->command_prevented = false;

	data->notify_sd = notify;

	ret = inc_wdt_establish_scc_inc_com(data, local_addr, remote_addr);
	if (ret) {
		fprintf(stderr, "Error initilize inc socket\n");
		return ret;
	}

	ret = pthread_create(&data->rx_task, NULL, inc_wdt_rx_task, data);
	if (ret)
		fprintf(stderr, "Error creating rx_task thread\n");

	return ret;
}

static int inc_wdt_check_wdt(struct inc_wdt_data *data, char *sysfs)
{
	int count = 0;
	FILE *fptr;

	fptr = fopen(sysfs, "r");
	if (!fptr) {
		fprintf(stderr, "Failed to read wdt sysfs interface\n");
		return -ENODEV;
	}

	fscanf(fptr,"%d",&count);
	fclose(fptr);

	if (data->ping_count == count)
		return -EINVAL;

	data->ping_count = count;

	return 0;
}

static void print_usage()
{
	printf("Usage: inc_wdt: [h] -r remote_addr -l local_addr -t timer_interval -k retry_interval -f ping_count_path\n");
}

int main(int argc, char *argv[])
{
	int ret = 0;
	struct inc_wdt_data *data;
	char sysfs_ping_count[MAX_PING_COUNT_PATH_LENGTH] = DEFAULT_SYSFS_PATH;
	char local_addr[MAX_ADDR_STRING_LENGTH] = DEFAULT_LOCAL_ADDR;
	char remote_addr[MAX_ADDR_STRING_LENGTH] = DEFAULT_REMOTE_ADDR;
	int option = 0;
	uint64_t usec = DEFAULT_TIMEVAL_USEC;
	uint64_t usec_retry = 0;
	bool notify = false;
	FILE *fp;

	while ((option = getopt(argc, argv,"hl:r:f:t:k:")) != -1) {
		switch (option) {
			case 'k':
				usec_retry = atol(optarg);
				break;
			case 'l':
				strncpy(local_addr, optarg,
					MAX_ADDR_STRING_LENGTH);
				break;
			case 'r':
				strncpy(remote_addr, optarg,
					MAX_ADDR_STRING_LENGTH);
				break;
			case 'f':
				strncpy(sysfs_ping_count, optarg,
					MAX_PING_COUNT_PATH_LENGTH);
				break;
			case 't':
				usec = atol(optarg);
				break;
			case 'h':
			default:
				print_usage();
				return 1;
		}
	}

	fp = fopen(sysfs_ping_count, "r");
	if (!fp) {
		fprintf(stderr, "Failed to open wdt ping count sysfs file %s\n",
			sysfs_ping_count);
		return 1;
	}
	fclose(fp);

#if HAVE_SYSTEMD
	ret = sd_watchdog_enabled(0, &usec);
	if (ret < 0) {
		fprintf(stderr, "Failed to get watchdog timeout value %d\n",
			ret);
		return 1;
	} else if (ret > 0) {
		/* send keep-alive message with half of WATCHDOG_USEC */
		usec = usec / 2;
		notify = true;
	}
#endif

	if (!usec_retry)
		usec_retry = usec / 3;


	printf("sysfs_ping_count: %s\nremote_addr: %s\nlocal_addr: %s\ntimeval: %" PRIu64 " usec\nretry time: %" PRIu64 " usec\n",
		sysfs_ping_count, remote_addr, local_addr, usec, usec_retry);

	printf("notify systemd: %s\n", notify ? "yes" : "no");

	data = calloc(1, sizeof(*data));
	if (!data) {
		fprintf(stderr, "failed to allocate memory\n");
		return 1;
	}

	ret = pthread_mutex_init(&data->lock, NULL);
	if (ret) {
		fprintf(stderr, "failed to initialize mutex\n");
		return 1;
	}

	ret = inc_wdt_init(data, local_addr, remote_addr, usec,
			   usec_retry, notify);
	if (ret) {
		fprintf(stderr, "failed to initialize inc_wdt\n");
		return 1;
	}

	do {
		struct timespec *timer;

		ret = inc_wdt_check_wdt(data, sysfs_ping_count);
		if (ret)
			fprintf(stderr, "Ping count of imx2 watchdog is not increasing\n");
		else {
			pthread_mutex_lock(&data->lock);
			ret = inc_wdt_send_wdg_msg(data);
			pthread_mutex_unlock(&data->lock);
			if (ret < 0)
				fprintf(stderr, "failed to send watchdog message\n");
		}

		if (ret < 0)
			timer = &data->timer_retry;
		else
			timer = &data->timer;

		clock_nanosleep(CLOCK_REALTIME, 0, timer, NULL);
	} while(1);

	return 0;
}
